import React, { useState, useEffect } from 'react';
import GatewayCard from './components/GatewayCard';
import ToggleSwitch from './components/ToggleSwitch';
import './App.css';

function App() {
  const [gateways, setGateways] = useState([]);
  const [loading, setLoading] = useState(true);
  const [theme, setTheme] = useState('light');

  useEffect(() => {
    loadGateways();
    loadTheme();
  }, []);

  const loadTheme = () => {
    const savedTheme = localStorage.getItem('irandargah_theme') || 'light';
    setTheme(savedTheme);
    document.documentElement.setAttribute('data-theme', savedTheme);
  };

  const loadGateways = async () => {
    try {
      const response = await fetch(`${window.irandargahData.restUrl}get-gateways`, {
        headers: {
          'X-WP-Nonce': window.irandargahData.restNonce
        }
      });
      const data = await response.json();
      if (data.success) {
        setGateways(data.gateways);
      }
    } catch (error) {
      console.error('Error loading gateways:', error);
    } finally {
      setLoading(false);
    }
  };

  const toggleTheme = () => {
    const newTheme = theme === 'light' ? 'dark' : 'light';
    setTheme(newTheme);
    localStorage.setItem('irandargah_theme', newTheme);
    document.documentElement.setAttribute('data-theme', newTheme);
  };

  const handleToggle = async (gatewayId, enabled) => {
    try {
      const response = await fetch(`${window.irandargahData.restUrl}toggle-gateway`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-WP-Nonce': window.irandargahData.restNonce
        },
        body: JSON.stringify({
          gateway_id: gatewayId,
          enabled: enabled
        })
      });
      const data = await response.json();
      if (data.success) {
        setGateways(prev => prev.map(g => 
          g.id === gatewayId ? { ...g, enabled: enabled ? 'yes' : 'no' } : g
        ));
      }
    } catch (error) {
      console.error('Error toggling gateway:', error);
    }
  };

  if (loading) {
    return <div className="irandargah-loading">در حال بارگذاری...</div>;
  }

  return (
    <div className="irandargah-dashboard-wrap" data-theme={theme}>
      <div className="irandargah-header">
        <div className="irandargah-header-content">
          <h1 className="irandargah-title">
            <svg className="irandargah-title-icon" width="32" height="32" viewBox="0 0 32 32" fill="none">
              <path d="M16 0C7.163 0 0 7.163 0 16s7.163 16 16 16 16-7.163 16-16S24.837 0 16 0zm0 29c-7.18 0-13-5.82-13-13S8.82 3 16 3s13 5.82 13 13-5.82 13-13 13zm-2-20v10l7 4.5L22 19l-6-3.75V9h-4z" fill="currentColor"/>
            </svg>
            درگاه‌های پرداخت ایران
          </h1>
          <div className="irandargah-header-actions">
            <button className="irandargah-theme-toggle" onClick={toggleTheme}>
              <span className="theme-icon-light">☀️</span>
              <span className="theme-icon-dark">🌙</span>
            </button>
          </div>
        </div>
      </div>

      <div className="irandargah-gateways-grid">
        {gateways.map(gateway => (
          <GatewayCard
            key={gateway.id}
            gateway={gateway}
            onToggle={handleToggle}
          />
        ))}
      </div>
    </div>
  );
}

export default App;


