<?php
defined('ABSPATH') || exit;

class IranDargah_Settings_Loader {
    
    /**
     * Get enabled status for a gateway
     * Checks irandargah_gateway_statuses first (source of truth), then falls back to WooCommerce settings
     * 
     * @param string $gateway_id Gateway ID
     * @param array $settings WooCommerce gateway settings
     * @return string 'yes' or 'no'
     */
    private static function get_gateway_enabled_status($gateway_id, $settings = []) {
        $statuses = get_option('irandargah_gateway_statuses', []);
        
        // Check irandargah_gateway_statuses first (source of truth)
        if (isset($statuses[$gateway_id])) {
            return (int)$statuses[$gateway_id] === 1 ? 'yes' : 'no';
        }
        
        // Fall back to WooCommerce settings
        return isset($settings['enabled']) ? $settings['enabled'] : 'no';
    }
    
    public static function get_all_gateways_data() {
        $gateways = [
            self::get_sep_gateway_data(),
            self::get_zarinpal_gateway_data(),
            self::get_aqayepardakht_gateway_data(),
            self::get_payping_gateway_data(),
            self::get_irankish_gateway_data(),
            self::get_parsian_gateway_data(),
            self::get_pasargad_gateway_data(),
            self::get_melli_gateway_data(),
            self::get_mellat_gateway_data(),
            self::get_zibal_gateway_data(),
        ];

        // Add extra gateways (Gateland gateways)
        $extra_gateways = self::get_extra_gateways_data();
        $gateways = array_merge($gateways, $extra_gateways);

        return array_filter($gateways); // حذف درگاه‌هایی که null برگردانده‌اند
    }
    
    /**
     * لیست پیکربندی درگاه‌های اضافی (Gateland) - مشترک بین get_extra_gateways_data و همگام‌سازی با Gateland
     */
    public static function get_extra_gateways_config() {
        return [
            'sepal' => [
                'title' => 'سپال',
                'description' => 'sepal.ir',
                'class' => 'Nabik\Gateland\Gateways\SepalGateway',
                'options' => [['label' => 'کلید وب سرویس', 'key' => 'api_key'], ['label' => 'هاست خارج از ایران', 'key' => 'non-iran-host', 'type' => 'checkbox']]
            ],
            'shepa' => [
                'title' => 'شپا',
                'description' => 'shepa.com',
                'class' => 'Nabik\Gateland\Gateways\ShepaGateway',
                'options' => [['label' => 'کلید api', 'key' => 'api']]
            ],
            'directpay' => [
                'title' => 'دایرکت پی',
                'description' => 'directpay.click',
                'class' => 'Nabik\Gateland\Gateways\DirectPayGateway',
                'options' => [['label' => 'Gateway ID', 'key' => 'gateway_id'], ['label' => 'Encryption Key', 'key' => 'encryption_key']]
            ],
            'jibit' => [
                'title' => 'جیبیت',
                'description' => 'jibit.ir',
                'class' => 'Nabik\Gateland\Gateways\JibitGateway',
                'options' => [['label' => 'API Key', 'key' => 'api_key'], ['label' => 'Secret Key', 'key' => 'secret_key']]
            ],
            'paystar' => [
                'title' => 'پی استار',
                'description' => 'paystar.ir',
                'class' => 'Nabik\Gateland\Gateways\PayStarGateway',
                'options' => [['label' => 'Gateway ID', 'key' => 'gateway_id'], ['label' => 'Encryption Key', 'key' => 'encryption_key']]
            ],
            'asanpardakht' => [
                'title' => 'آسان پرداخت',
                'description' => 'asanpardakht.ir',
                'class' => 'Nabik\Gateland\Gateways\AsanPardakhtGateway',
                'options' => [['label' => 'Merchant Configuration ID', 'key' => 'merchant']]
            ],
            'nabikpay' => [
                'title' => 'نابیک پی',
                'description' => 'اتصال به نابیک‌پی',
                'class' => 'Nabik\Gateland\Gateways\NabikPayGateway',
                'options' => [['label' => 'آدرس نابیک‌پی', 'key' => 'url'], ['label' => 'کلید مرچنت', 'key' => 'merchant']]
            ],
            'sizpay' => [
                'title' => 'سیز پی',
                'description' => 'sizpay.ir',
                'class' => 'Nabik\Gateland\Gateways\SizPayGateway',
                'options' => [['label' => 'کلید', 'key' => 'key']]
            ],
            'vandar' => [
                'title' => 'وندار',
                'description' => 'vandar.io',
                'class' => 'Nabik\Gateland\Gateways\VandarGateway',
                'options' => [['label' => 'API Key', 'key' => 'api_key']]
            ],
        ];
    }

    /**
     * همگام‌سازی درگاه اضافی با جدول Gateland تا در تسویه‌حساب ووکامرس نمایش داده شود
     */
    public static function sync_extra_gateway_to_gateland($gateway_key) {
        if (!class_exists('Nabik\Gateland\Models\Gateway') || !class_exists('Nabik_Net_Database')) {
            return;
        }
        $configs = self::get_extra_gateways_config();
        if (!isset($configs[$gateway_key])) {
            return;
        }
        $config = $configs[$gateway_key];
        $option_name = 'irandargah_extra_' . $gateway_key . '_settings';
        $settings = get_option($option_name, []);
        $enabled = isset($settings['enabled']) && $settings['enabled'] === 'yes';

        try {
            if (!\Nabik_Net_Database::Schema()->hasTable('gateland_gateways')) {
                if (class_exists('Nabik\Gateland\Install')) {
                    \Nabik\Gateland\Install::create_tables();
                }
            }
        } catch (\Throwable $e) {
            return;
        }

        $map = get_option('irandargah_extra_gateway_id_map', []);
        $extra_id = 'extra_' . $gateway_key;
        $existing_gateland_id = isset($map[$extra_id]) ? (int) $map[$extra_id] : null;

        $data = [];
        foreach ($config['options'] as $opt) {
            $key = $opt['key'];
            if (isset($settings[$key])) {
                $data[$key] = $settings[$key];
            }
        }
        if (isset($settings['sandbox'])) {
            $data['sandbox'] = $settings['sandbox'];
        }

        if ($enabled) {
            $currencies = ['IRT'];
            $status = 'active';
            if ($existing_gateland_id) {
                $gateway = \Nabik\Gateland\Models\Gateway::query()->find($existing_gateland_id);
                if ($gateway) {
                    $gateway->status = $status;
                    $gateway->data = $data;
                    $gateway->currencies = $currencies;
                    $gateway->save();
                }
            } else {
                $gateway = \Nabik\Gateland\Models\Gateway::query()
                    ->where('class', $config['class'])
                    ->first();
                if ($gateway) {
                    $gateway->status = $status;
                    $gateway->data = $data;
                    $gateway->currencies = $currencies;
                    $gateway->save();
                    $map[$extra_id] = $gateway->id;
                    update_option('irandargah_extra_gateway_id_map', $map);
                } else {
                    $max_sort = \Nabik\Gateland\Models\Gateway::query()->max('sort');
                    $gateway = new \Nabik\Gateland\Models\Gateway();
                    $gateway->class = $config['class'];
                    $gateway->status = $status;
                    $gateway->sort = is_numeric($max_sort) ? (int) $max_sort + 1 : 0;
                    $gateway->data = $data;
                    $gateway->currencies = $currencies;
                    $gateway->save();
                    $map[$extra_id] = (int) $gateway->id;
                    update_option('irandargah_extra_gateway_id_map', $map);
                }
            }
        } else {
            if ($existing_gateland_id) {
                $gateway = \Nabik\Gateland\Models\Gateway::query()->find($existing_gateland_id);
                if ($gateway) {
                    $gateway->status = 'inactive';
                    $gateway->save();
                }
            } else {
                $gateway = \Nabik\Gateland\Models\Gateway::query()->where('class', $config['class'])->first();
                if ($gateway) {
                    $gateway->status = 'inactive';
                    $gateway->save();
                }
            }
        }

        // فقط یک ردیف فعال برای هر کلاس درگاه (جلوگیری از نمایش تکراری در تسویه‌حساب)
        if ($enabled && isset($gateway) && $gateway && $gateway->id) {
            \Nabik\Gateland\Models\Gateway::query()
                ->where('class', $config['class'])
                ->where('id', '!=', $gateway->id)
                ->update(['status' => 'inactive']);
        }

        delete_transient('gateland_active_gateways');
        if (class_exists('Nabik\Gateland\Services\GatewayService') && method_exists('Nabik\Gateland\Services\GatewayService', 'reset_activated')) {
            \Nabik\Gateland\Services\GatewayService::reset_activated();
        }
    }

    /**
     * همگام‌سازی همه درگاه‌های اضافی فعال با Gateland (برای نمایش در تسویه‌حساب)
     */
    public static function sync_all_extra_gateways_to_gateland() {
        static $synced = false;
        if ($synced) {
            return;
        }
        $synced = true;
        $configs = self::get_extra_gateways_config();
        foreach (array_keys($configs) as $gateway_key) {
            $settings = get_option('irandargah_extra_' . $gateway_key . '_settings', []);
            if (isset($settings['enabled']) && $settings['enabled'] === 'yes') {
                self::sync_extra_gateway_to_gateland($gateway_key);
            }
        }
    }

    /**
     * Get all extra gateways (Gateland) data
     */
    public static function get_extra_gateways_data() {
        $gateways_config = self::get_extra_gateways_config();
        $gateways = [];
        
        foreach ($gateways_config as $gateway_key => $config) {
            $settings_option = 'irandargah_extra_' . $gateway_key . '_settings';
            $settings = get_option($settings_option, []);
            
            $gateway = [
                'id' => 'extra_' . $gateway_key,
                'title' => $config['title'],
                'description' => $config['description'],
                'icon' => self::get_generic_icon_svg(),
                'enabled' => $settings['enabled'] ?? 'no',
                'sandbox' => $settings['sandbox'] ?? 'no',
                'settings_sections' => [
                    'basic' => [
                        'title' => __('تنظیمات پایه', 'irandargah'),
                        'fields' => [
                            'title' => [
                                'title' => __('عنوان درگاه', 'irandargah'),
                                'type' => 'text',
                                'value' => $settings['title'] ?? $config['title'],
                            ],
                            'description' => [
                                'title' => __('توضیحات', 'irandargah'),
                                'type' => 'textarea',
                                'value' => $settings['description'] ?? '',
                            ],
                        ],
                    ],
                    'account' => [
                        'title' => __('تنظیمات حساب', 'irandargah'),
                        'fields' => array_merge(
                            array_reduce($config['options'], function($fields, $option) use ($settings, $gateway_key) {
                                $key = $option['key'];
                                $fields[$key] = [
                                    'title' => $option['label'],
                                    'type' => isset($option['type']) && $option['type'] === 'checkbox' ? 'checkbox' : 'text',
                                    'value' => $settings[$key] ?? '',
                                    'label' => $option['label'] ?? '',
                                    'description' => $option['description'] ?? '',
                                ];
                                return $fields;
                            }, []),
                            [
                                'sandbox' => [
                                    'title' => __('حالت آزمایشی', 'irandargah'),
                                    'type' => 'checkbox',
                                    'value' => $settings['sandbox'] ?? 'no',
                                    'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                                    'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                                ],
                            ]
                        ),
                    ],
                ],
            ];
            
            $gateways[] = $gateway;
        }
        
        return $gateways;
    }
    
    public static function get_sep_gateway_data() {
        $settings = get_option('woocommerce_WC_Sep_Payment_Gateway_settings', []);
        $accounts = get_option('woocommerce_sep_accounts', []);
        
        $callback_url = WC()->api_request_url('WC_Sep_Payment_Gateway');
        
        return [
            'id' => 'WC_Sep_Payment_Gateway',
            'title' => __('بانک سامان', 'irandargah'),
            'description' => __('درگاه پرداخت بانک سامان برای ووکامرس', 'irandargah'),
            'icon' => self::get_sep_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('WC_Sep_Payment_Gateway', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('بانک سامان', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'merchantcode' => [
                            'title' => __('کد پذیرنده (Terminal ID)', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['merchantcode'] ?? '',
                            'description' => __('کد پذیرنده دریافتی از بانک سامان', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل بانک سامان تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('پرداخت با موفقیت انجام شد.', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت ناموفق بود.', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }
    
    public static function get_zarinpal_gateway_data() {
        $settings = get_option('woocommerce_WC_ZPal_settings', []);
        
        $callback_url = WC()->api_request_url('WC_ZPal');
        
        return [
            'id' => 'WC_ZPal',
            'title' => __('زرین‌پال', 'irandargah'),
            'description' => __('درگاه پرداخت امن زرین‌پال', 'irandargah'),
            'icon' => self::get_zarinpal_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('WC_ZPal', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('پرداخت امن زرین‌پال', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'merchantcode' => [
                            'title' => __('مرچنت کد', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['merchantcode'] ?? '',
                            'description' => __('مرچنت کد دریافتی از زرین‌پال', 'irandargah'),
                            'placeholder' => __('مثال: 12345678-1234-1234-1234-123456789012', 'irandargah'),
                        ],
                        'access_token' => [
                            'title' => __('توکن دسترسی', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['access_token'] ?? '',
                            'description' => __('توکن دسترسی برای سرویس استرداد وجه (اختیاری)', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل زرین‌پال تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('پرداخت با موفقیت انجام شد.', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت ناموفق بود.', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    private static function get_sep_icon_svg() {
        return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <path d="M12 2L2 7l10 5 10-5-10-5z"/>
            <path d="M2 17l10 5 10-5"/>
            <path d="M2 12l10 5 10-5"/>
        </svg>';
    }

    private static function get_zarinpal_icon_svg() {
        return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <circle cx="12" cy="12" r="10"/>
            <path d="M12 6v6l4 2"/>
        </svg>';
    }

    public static function get_generic_icon_svg() {
        return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <rect x="1" y="4" width="22" height="16" rx="2" ry="2"/>
            <line x1="1" y1="10" x2="23" y2="10"/>
        </svg>';
    }

    public static function get_aqayepardakht_gateway_data() {
        $settings = get_option('woocommerce_WC_aqayepardakht_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_aqayepardakht');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_aqayepardakht',
            'title' => __('آقای پرداخت', 'irandargah'),
            'description' => __('درگاه پرداخت آقای پرداخت برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('WC_aqayepardakht', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('آقای پرداخت', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'api' => [
                            'title' => __('API Key', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['api'] ?? '',
                            'description' => __('کلید API دریافتی از آقای پرداخت', 'irandargah'),
                            'placeholder' => __('مثال: xxxxx-xxxxx-xxxxx', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل آقای پرداخت تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_payping_gateway_data() {
        $settings = get_option('woocommerce_WC_payping_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_payping');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_payping',
            'title' => __('پی‌پینگ', 'irandargah'),
            'description' => __('درگاه پرداخت پی‌پینگ برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('WC_payping', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('پرداخت از طریق پی‌پینگ', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'paypingToken' => [
                            'title' => __('توکن', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['paypingToken'] ?? '',
                            'description' => __('توکن درگاه پی‌پینگ', 'irandargah'),
                            'placeholder' => __('مثال: xxxxx-xxxxx-xxxxx', 'irandargah'),
                        ],
                        'ioserver' => [
                            'title' => __('سرور خارج', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['ioserver'] ?? 'no',
                            'label' => __('اتصال به سرور خارج', 'irandargah'),
                            'description' => __('در صورت تیک خوردن، درگاه به سرور خارج از کشور متصل می‌شود', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('در حالت آزمایشی تراکنش واقعی انجام نمی‌شود و به صفحه تست داخلی هدایت می‌شود.', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل پی‌پینگ تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_irankish_gateway_data() {
        $settings = get_option('woocommerce_irankish_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Gateway_Irankish');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'irankish',
            'title' => __('ایران‌کیش', 'irandargah'),
            'description' => __('درگاه پرداخت ایران‌کیش برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('irankish', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('پرداخت امن ایران کیش', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'terminalID' => [
                            'title' => __('شماره ترمینال', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['terminalID'] ?? '',
                            'description' => __('شماره ترمینال درگاه ایران‌کیش', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'password' => [
                            'title' => __('کلمه عبور', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['password'] ?? '',
                            'description' => __('کلمه عبور درگاه ایران‌کیش', 'irandargah'),
                        ],
                        'acceptOrId' => [
                            'title' => __('شناسه پذیرنده', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['acceptOrId'] ?? '',
                            'description' => __('شناسه پذیرنده درگاه ایران‌کیش', 'irandargah'),
                        ],
                        'pubKey' => [
                            'title' => __('کلید عمومی', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pubKey'] ?? '',
                            'description' => __('کلید عمومی درگاه ایران‌کیش', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل ایران‌کیش تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_parsian_gateway_data() {
        $settings = get_option('woocommerce_bankparsian_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Gateway_Bankparsian');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'bankparsian',
            'title' => __('بانک پارسیان', 'irandargah'),
            'description' => __('درگاه پرداخت بانک پارسیان برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('bankparsian', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('پرداخت امن پارسیان', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'pin' => [
                            'title' => __('PIN', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['pin'] ?? '',
                            'description' => __('PIN دریافتی از بانک پارسیان', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل بانک پارسیان تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_pasargad_gateway_data() {
        $settings = get_option('woocommerce_pasargad_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Pasargad_Gateway');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'pasargad',
            'title' => __('بانک پاسارگاد', 'irandargah'),
            'description' => __('درگاه پرداخت بانک پاسارگاد برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('pasargad', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('بانک پاسارگاد', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'pasargad_terminal_id' => [
                            'title' => __('شماره ترمینال', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pasargad_terminal_id'] ?? '',
                            'description' => __('شماره ترمینال درگاه پاسارگاد', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'pasargad_merchant_id' => [
                            'title' => __('شماره فروشگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pasargad_merchant_id'] ?? '',
                            'description' => __('شماره فروشگاه درگاه پاسارگاد', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'pasargad_username' => [
                            'title' => __('نام کاربری', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pasargad_username'] ?? '',
                            'description' => __('نام کاربری درگاه پاسارگاد', 'irandargah'),
                        ],
                        'pasargad_password' => [
                            'title' => __('کلمه عبور', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['pasargad_password'] ?? '',
                            'description' => __('کلمه عبور درگاه پاسارگاد', 'irandargah'),
                        ],
                        'base_url' => [
                            'title' => __('آدرس پایه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['base_url'] ?? 'https://pep.shaparak.ir/pepg',
                            'description' => __('آدرس پایه درگاه پاسارگاد', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'is_finish_order' => [
                            'title' => __('وضعیت سفارش پس از پرداخت', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['is_finish_order'] ?? 'no',
                            'label' => __('سفارش به وضعیت تکمیل تغییر یابد', 'irandargah'),
                            'description' => __('در صورت فعال بودن، سفارش به وضعیت تکمیل تغییر می‌یابد، در غیر این صورت به وضعیت در حال آماده‌سازی', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل پاسارگاد تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} و {SaleOrderId} برای نمایش کد رهگیری و شماره درخواست استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_melli_gateway_data() {
        $settings = get_option('woocommerce_melli_pay_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Gateway_Melli');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'melli_pay',
            'title' => __('بانک ملّی', 'irandargah'),
            'description' => __('درگاه پرداخت بانک ملّی برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('melli_pay', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('بانک ملی', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'terminal_id' => [
                            'title' => __('شماره ترمینال', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['terminal_id'] ?? '',
                            'description' => __('شماره ترمینال درگاه بانک ملی', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'merchant_id' => [
                            'title' => __('شماره پذیرنده', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['merchant_id'] ?? '',
                            'description' => __('شماره پذیرنده درگاه بانک ملی', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'terminal_key' => [
                            'title' => __('کلید ترمینال', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['terminal_key'] ?? '',
                            'description' => __('کلید ترمینال درگاه بانک ملی', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل بانک ملی تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_mellat_gateway_data() {
        $settings = get_option('woocommerce_BankMellat_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_BankMellat');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'BankMellat',
            'title' => __('بانک ملت', 'irandargah'),
            'description' => __('درگاه پرداخت بانک ملت برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('BankMellat', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('بانک ملت', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'terminal' => [
                            'title' => __('ترمینال آیدی', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['terminal'] ?? '',
                            'description' => __('شماره ترمینال درگاه بانک ملت', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'username' => [
                            'title' => __('نام کاربری', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['username'] ?? '',
                            'description' => __('نام کاربری درگاه بانک ملت', 'irandargah'),
                        ],
                        'password' => [
                            'title' => __('کلمه عبور', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['password'] ?? '',
                            'description' => __('کلمه عبور درگاه بانک ملت', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('در حالت آزمایشی تراکنش واقعی انجام نمی‌شود و به صفحه تست داخلی هدایت می‌شود.', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل بانک ملت تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} و {SaleOrderId} برای نمایش کد رهگیری و شماره درخواست استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_zibal_gateway_data() {
        $settings = get_option('woocommerce_WC_Zibal_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Zibal');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_Zibal',
            'title' => __('زیبال', 'irandargah'),
            'description' => __('درگاه پرداخت زیبال برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => self::get_gateway_enabled_status('WC_Zibal', $settings),
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('زیبال', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'pin' => [
                            'title' => __('مرچنت کد', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pin'] ?? '',
                            'description' => __('مرچنت کد دریافتی از زیبال', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل زیبال تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function save_gateway_settings($gateway_id, $settings) {
        // Check if this is an extra gateway
        if (strpos($gateway_id, 'extra_') === 0) {
            // Handle extra gateways
            $gateway_key = str_replace('extra_', '', $gateway_id);
            $option_name = 'irandargah_extra_' . $gateway_key . '_settings';
            $current_settings = get_option($option_name, []);
            
            foreach ($settings as $key => $value) {
                if ($key === 'enabled' || $key === 'sandbox' || $key === 'non-iran-host') {
                    $current_settings[$key] = $value === 'yes' ? 'yes' : 'no';
                } elseif ($key === 'description') {
                    $current_settings[$key] = sanitize_textarea_field($value);
                } else {
                    $current_settings[$key] = sanitize_text_field($value);
                }
            }
            
            update_option($option_name, $current_settings);
            $gateway_key = str_replace('extra_', '', $gateway_id);
            self::sync_extra_gateway_to_gateland($gateway_key);
            return true;
        }
        
        // Handle regular gateways
        $option_name = "woocommerce_{$gateway_id}_settings";
        $current_settings = get_option($option_name, []);
        
        foreach ($settings as $key => $value) {
            if ($key === 'enabled' || $key === 'sandbox' || in_array($key, ['ioserver', 'multi', 'is_finish_order'], true)) {
                $current_settings[$key] = $value === 'yes' ? 'yes' : 'no';
            } 
            elseif (in_array($key, ['fee_payer', 'order_status', 'vahed'], true)) {
                $current_settings[$key] = sanitize_text_field($value);
            } 
            elseif (in_array($key, [
                'title', 'description', 'merchantcode', 'terminal', 'terminal_id', 'merchant_id', 
                'terminal_key', 'pin', 'pasargad_terminal_id', 'pasargad_merchant_id', 
                'pasargad_username', 'pasargad_password', 'base_url', 'username', 'password',
                'terminalID', 'acceptOrId', 'pubKey', 'paypingToken', 'api', 'api_key', 'access_token',
                'gateway_id', 'encryption_key', 'secret_key', 'merchant', 'url', 'key'
            ], true)) {
                $current_settings[$key] = sanitize_text_field($value);
            } 
            elseif (in_array($key, [
                'success_massage', 'failed_massage', 'cancelled_massage', 
                'success_message', 'failed_message', 'canceled_message',
                'connecting_message', 'connection_error_massage'
            ], true)) {
                $current_settings[$key] = sanitize_textarea_field($value);
            } 
            else {
                $current_settings[$key] = sanitize_text_field($value);
            }
        }
        
        update_option($option_name, $current_settings);
        
        return true;
    }

    public static function toggle_gateway($gateway_id, $enabled) {
        // Check if this is an extra gateway
        if (strpos($gateway_id, 'extra_') === 0) {
            // Handle extra gateways
            $gateway_key = str_replace('extra_', '', $gateway_id);
            $option_name = 'irandargah_extra_' . $gateway_key . '_settings';
            $gateway_settings = get_option($option_name, []);
            $gateway_settings['enabled'] = $enabled ? 'yes' : 'no';
            update_option($option_name, $gateway_settings);
            self::sync_extra_gateway_to_gateland($gateway_key);
            return true;
        }
        
        // Handle regular gateways
        $statuses = get_option('irandargah_gateway_statuses', []);
        $statuses[$gateway_id] = $enabled ? 1 : 0;
        update_option('irandargah_gateway_statuses', $statuses);

        $option_name = "woocommerce_{$gateway_id}_settings";
        $gateway_settings = get_option($option_name, []);
        $gateway_settings['enabled'] = $enabled ? 'yes' : 'no';
        update_option($option_name, $gateway_settings);
        
        return true;
    }
}
