<?php
defined('ABSPATH') || exit;

class IranDargah_REST_Endpoints {
    
    public static function init() {
        add_action('rest_api_init', [__CLASS__, 'register_routes']);
    }
    
    public static function register_routes() {
        register_rest_route('irandargah/v1', '/toggle-gateway', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'toggle_gateway'],
            'permission_callback' => [__CLASS__, 'check_permission'],
        ]);
        
        register_rest_route('irandargah/v1', '/save-settings', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'save_settings'],
            'permission_callback' => [__CLASS__, 'check_permission'],
        ]);
        
        register_rest_route('irandargah/v1', '/get-settings', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_settings'],
            'permission_callback' => [__CLASS__, 'check_permission'],
        ]);
        
        register_rest_route('irandargah/v1', '/get-gateways', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_gateways'],
            'permission_callback' => [__CLASS__, 'check_permission'],
        ]);
        
        register_rest_route('irandargah/v1', '/get-alerts', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_alerts'],
            'permission_callback' => [__CLASS__, 'check_permission'],
        ]);
        
        register_rest_route('irandargah/v1', '/get-alerts-summary', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_alerts_summary'],
            'permission_callback' => [__CLASS__, 'check_permission'],
        ]);
        
        register_rest_route('irandargah/v1', '/get-extra-gateways', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_extra_gateways'],
            'permission_callback' => [__CLASS__, 'check_permission'],
        ]);
        
        register_rest_route('irandargah/v1', '/toggle-extra-gateway', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'toggle_extra_gateway'],
            'permission_callback' => [__CLASS__, 'check_permission'],
        ]);
        
        register_rest_route('irandargah/v1', '/save-extra-settings', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'save_extra_settings'],
            'permission_callback' => [__CLASS__, 'check_permission'],
        ]);
    }
    
    public static function check_permission() {
        return current_user_can('manage_options');
    }
    
    public static function toggle_gateway($request) {
        $gateway_id = sanitize_text_field($request->get_param('gateway_id'));
        $enabled_param = $request->get_param('enabled');
        
        // Handle different formats: boolean true/false, string 'true'/'false', string 'yes'/'no'
        $enabled = false;
        if ($enabled_param === true || $enabled_param === 'true' || $enabled_param === 'yes' || $enabled_param === 1 || $enabled_param === '1') {
            $enabled = true;
        }
        
        if (empty($gateway_id)) {
            return new WP_Error('invalid_gateway', __('شناسه درگاه معتبر نیست.', 'irandargah'), ['status' => 400]);
        }
        
        IranDargah_Settings_Loader::toggle_gateway($gateway_id, $enabled);
        
        // Verify the change was saved
        $statuses = get_option('irandargah_gateway_statuses', []);
        $saved_status = isset($statuses[$gateway_id]) ? (int)$statuses[$gateway_id] : 0;
        
        return new WP_REST_Response([
            'success' => true,
            'message' => __('وضعیت درگاه تغییر کرد.', 'irandargah'),
            'enabled' => $saved_status === 1,
        ], 200);
    }
    
    public static function save_settings($request) {
        $gateway_id = sanitize_text_field($request->get_param('gateway_id'));
        $settings = $request->get_param('settings');
        
        if (empty($gateway_id) || !is_array($settings)) {
            return new WP_Error('invalid_data', __('داده‌های ارسالی معتبر نیست.', 'irandargah'), ['status' => 400]);
        }
        
        IranDargah_Settings_Loader::save_gateway_settings($gateway_id, $settings);
        
        return new WP_REST_Response([
            'success' => true,
            'message' => __('تنظیمات با موفقیت ذخیره شد.', 'irandargah'),
        ], 200);
    }
    
    public static function get_settings($request) {
        $gateway_id = sanitize_text_field($request->get_param('gateway_id'));
        
        if (empty($gateway_id)) {
            return new WP_Error('invalid_gateway', __('شناسه درگاه معتبر نیست.', 'irandargah'), ['status' => 400]);
        }
        
        $option_name = "woocommerce_{$gateway_id}_settings";
        $settings = get_option($option_name, []);
        
        return new WP_REST_Response([
            'success' => true,
            'settings' => $settings,
        ], 200);
    }
    
    public static function get_gateways($request) {
        if (!class_exists('IranDargah_Settings_Loader')) {
            require_once dirname(__FILE__) . '/settings-loader.php';
        }
        
        $gateways = IranDargah_Settings_Loader::get_all_gateways_data();
        
        return new WP_REST_Response([
            'success' => true,
            'gateways' => $gateways ?: [],
        ], 200);
    }
    
    public static function get_alerts($request) {
        if (!class_exists('IranDargah_Gateway_Alerts')) {
            require_once dirname(__FILE__) . '/gateway-alerts.php';
        }
        
        $gateway_id = $request->get_param('gateway_id');
        
        if ($gateway_id) {
            // Get alerts for specific gateway
            $alerts = IranDargah_Gateway_Alerts::get_gateway_alerts($gateway_id);
            
            return new WP_REST_Response([
                'success' => true,
                'alerts' => $alerts,
            ], 200);
        }
        
        // Get all alerts
        $all_alerts = IranDargah_Gateway_Alerts::get_all_alerts();
        
        return new WP_REST_Response([
            'success' => true,
            'alerts' => $all_alerts,
        ], 200);
    }
    
    public static function get_alerts_summary($request) {
        if (!class_exists('IranDargah_Gateway_Alerts')) {
            require_once dirname(__FILE__) . '/gateway-alerts.php';
        }
        
        $summary = IranDargah_Gateway_Alerts::get_alerts_summary();
        
        return new WP_REST_Response([
            'success' => true,
            'summary' => $summary,
        ], 200);
    }
    
    public static function get_extra_gateways($request) {
        if (!class_exists('IranDargah_Settings_Loader')) {
            require_once dirname(__FILE__) . '/settings-loader.php';
        }
        
        $gateways = IranDargah_Settings_Loader::get_extra_gateways_data();
        
        return new WP_REST_Response([
            'success' => true,
            'gateways' => $gateways ?: [],
        ], 200);
    }
    
    public static function toggle_extra_gateway($request) {
        $gateway_id = sanitize_text_field($request->get_param('gateway_id'));
        $enabled = $request->get_param('enabled') === true || $request->get_param('enabled') === 'true' || $request->get_param('enabled') === 'yes';
        
        if (empty($gateway_id)) {
            return new WP_Error('invalid_gateway', __('شناسه درگاه معتبر نیست.', 'irandargah'), ['status' => 400]);
        }
        
        // Extract gateway key
        $gateway_key = str_replace('extra_', '', $gateway_id);
        $settings_option = 'irandargah_extra_' . $gateway_key . '_settings';
        
        // Get existing settings
        $settings = get_option($settings_option, []);
        
        // Update enabled status
        $settings['enabled'] = $enabled ? 'yes' : 'no';
        
        // Save settings
        update_option($settings_option, $settings);
        
        return new WP_REST_Response([
            'success' => true,
            'message' => __('وضعیت درگاه تغییر کرد.', 'irandargah'),
        ], 200);
    }
    
    public static function save_extra_settings($request) {
        $gateway_id = sanitize_text_field($request->get_param('gateway_id'));
        $settings = $request->get_param('settings');
        
        if (empty($gateway_id) || !is_array($settings)) {
            return new WP_Error('invalid_data', __('داده‌های ارسالی معتبر نیست.', 'irandargah'), ['status' => 400]);
        }
        
        // Extract gateway key
        $gateway_key = str_replace('extra_', '', $gateway_id);
        $settings_option = 'irandargah_extra_' . $gateway_key . '_settings';
        
        // Get existing settings
        $existing_settings = get_option($settings_option, []);
        
        // Merge with new settings
        $existing_settings = array_merge($existing_settings, $settings);
        
        // Save settings
        update_option($settings_option, $existing_settings);
        
        return new WP_REST_Response([
            'success' => true,
            'message' => __('تنظیمات با موفقیت ذخیره شد.', 'irandargah'),
        ], 200);
    }
}

IranDargah_REST_Endpoints::init();

